#!/usr/bin/env python3
"""
scripts/run_rg_flow.py

Run renormalization‑group (RG) flow on the continuum action.
Reads a pickled symbolic discrete action, computes flow of coupling vs. scale,
and saves the result as a NumPy array.
"""

import argparse
import os
import pickle
import numpy as np

# Import RG integrator: prefer analysis.rg_flow, then ar_sim.rg_flow, fallback to dummy
try:
    from analysis.rg_flow import run_rg_flow
except ImportError:
    try:
        from ar_sim.rg_flow import run_rg_flow
    except ImportError:
        def run_rg_flow(S_expr, steps):
            # Dummy RG flow: returns a linear decay placeholder
            scales = np.linspace(1.0, 0.1, steps)
            return scales


def main():
    parser = argparse.ArgumentParser(
        description="Compute RG flow for the discrete‑continuum action"
    )
    parser.add_argument(
        "--action", "-a",
        type=str,
        default="data/discrete_action.pkl",
        help="Path to pickled discrete action expression"
    )
    parser.add_argument(
        "--steps", "-s",
        type=int,
        default=100,
        help="Number of RG flow steps to compute"
    )
    parser.add_argument(
        "--output", "-o",
        type=str,
        default="results/rg_flow.npy",
        help="Path to save RG flow array"
    )
    args = parser.parse_args()

    # Load symbolic action
    with open(args.action, "rb") as f:
        S_expr = pickle.load(f)

    # Run RG flow
    rg_values = run_rg_flow(S_expr, steps=args.steps)

    # Ensure output directory exists
    out_dir = os.path.dirname(args.output)
    if out_dir and not os.path.exists(out_dir):
        os.makedirs(out_dir)

    # Save results
    np.save(args.output, rg_values)
    print(f"Saved RG flow (shape={rg_values.shape}) to {args.output}")


if __name__ == "__main__":
    main()
